<?php

namespace App\Http\Controllers\Api;

use App\Models\ProductType;
use App\Models\Product;
use App\Models\Category;
use App\Models\SubCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Api\BaseController;

class AdminProductController extends BaseController
{
    private $product;

    public function __construct(ProductType $product)
    {
        $this->middleware('auth:user_api'); // agar admin ke liye alag API guard use kar rahe ho
        $this->product = $product;
    }

    public function index()
    {
        try {
            $data = ProductType::with('category','subCategory')->get();
            return $this->respond($data, [], 200, 'Products fetched successfully');
        } catch (\Exception $e) {
            return $this->respond([], [], 500, $e->getMessage(), false);
        }
    }

    public function store(Request $request)
    {
        $request->validate([
            'name'          => 'required|string|max:255',
            'category_id'   => 'required|exists:categories,id',
            'sub_category_id' => 'nullable|exists:sub_categories,id',
            'colors'        => 'nullable',
            'icon'          => 'nullable|image|mimes:png,jpg,jpeg,svg',
            'size_chart'    => 'nullable|image|mimes:png,jpg,jpeg,svg',
        ]);

        try {
            DB::beginTransaction();

            $data = $request->except(['icon', 'size_chart']);

            // Icon upload
            if ($request->hasFile('icon')) {
                $data['icon'] = $request->file('icon')->store('products/icons', 'public');
            }

            // Size chart upload
            if ($request->hasFile('size_chart')) {
                $data['size_chart'] = $request->file('size_chart')->store('products/size_charts', 'public');
            }

            // Handle colors
            $data['colors'] = is_array($request->colors) ? $request->colors : explode(',', $request->colors ?? '');

            $product = $this->product->create($data);

            DB::commit();
            return $this->respond($product, [], 201, 'Product added successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->respond([], [], 500, $e->getMessage(), false);
        }
    }
    
    
    public function storeImages(Request $request, $id)
    {
        try {
            DB::beginTransaction();
    
            $product = Product::findOrFail($id);
    
            // Multiple images upload
            if ($request->hasFile('images')) {
                foreach ($request->file('images') as $image) {
                    $path = $image->store('products/images', 'public');
                    $product->images()->create([
                        'image' => $path,
                    ]);
                }
            }
            
            $product->status = 'Accepted';
            $product->save();
    
            DB::commit();
            return $this->respond($product->load('images'), [], 201, 'Product images added successfully');
    
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->respond([], [], 500, $e->getMessage(), false);
        }
    }

    public function show($id)
    {
        try {
            $data = ProductType::with('category','subCategory')->findOrFail($id);
            return $this->respond($data, [], 200, 'Product details fetched');
        } catch (\Exception $e) {
            return $this->respond([], [], 404, $e->getMessage(), false);
        }
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'name'          => 'nullable|string|max:255',
            'category_id'   => 'nullable|exists:categories,id',
            'sub_category_id' => 'nullable|exists:sub_categories,id',
            'colors'        => 'nullable',
            'icon'          => 'nullable|image|mimes:png,jpg,jpeg,svg',
            'size_chart'    => 'nullable|image|mimes:png,jpg,jpeg,svg',
        ]);

        try {
            DB::beginTransaction();

            $data = $request->except(['icon', 'size_chart']);

            // Icon replacement
            if ($request->hasFile('icon')) {
                $data['icon'] = $request->file('icon')->store('products/icons', 'public');
            }

            // Size chart replacement
            if ($request->hasFile('size_chart')) {
                $data['size_chart'] = $request->file('size_chart')->store('products/size_charts', 'public');
            }

            // Colors handling
            if ($request->has('colors')) {
                $data['colors'] = is_array($request->colors) ? $request->colors : explode(',', $request->colors);
            }

            $product = $this->product->findOrFail($id);
            $product->update($data);

            DB::commit();
            return $this->respond($product, [], 200, 'Product updated successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->respond([], [], 500, $e->getMessage(), false);
        }
    }

    public function destroy($id)
    {
        try {
            $product = $this->product->findOrFail($id);
            $product->delete();

            return $this->respond([], [], 200, 'Product deleted successfully');
        } catch (\Exception $e) {
            return $this->respond([], [], 500, $e->getMessage(), false);
        }
    }
}
