<?php

namespace App\Http\Controllers\Admin;

use App\Models\ProductType;
use App\Models\Category;
use App\Models\SubCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;

class ProductController extends Controller
{
    private $product;

    public function __construct(ProductType $product)
    {
        $this->middleware('auth:admin');
        $this->product = $product;
    }

    public function index()
    {
        try {
            $data = $this->product->all(); // ya ->allProducts() agar custom method banani ho
            return view('admin.products.index', compact('data'));
        } catch (\Exception $e) {
            session()->flash('error', $e->getMessage());
            return redirect()->back();
        }
    }

    public function create()
    {
        $data = new ProductType();
        $form = [
            'type' => 'create',
            'heading' => 'Add Product',
            'method' => 'POST',
            'action' => route('admin.products.store'),
            'cancel_url' => route('admin.products.index')
        ];
        
        $categories = Category::pluck('name', 'id');
        $subCategories = SubCategory::pluck('name', 'id');
        return view('admin.products.form', compact('data', 'form', 'categories', 'subCategories'));
    }

    public function store(Request $request)
    {
        try {
            DB::beginTransaction();
    
            $data = $request->except(['_method', '_token', 'id', 'icon']);
    
            // Handle icon upload
            if ($request->hasFile('icon')) {
                $data['icon'] = $request->file('icon')->store('products/icons', 'public');
            }
            
            if ($request->hasFile('size_chart')) {
                $data['size_chart'] = $request->file('size_chart')->store('products/icons', 'public');
            }
            
            // $im = [];
            // if ($request->images) {
            //     foreach ($request->images as $image) {
            //         $im[] = $image->store('products/icons', 'public');
            //     }
            // }
            // $data['images'] = json_encode($im);
    
            // Handle multiple colors (comma-separated or array)
            $data['colors'] = is_array($request->colors) ? $request->colors : explode(',', $request->colors);
            
            if ($request->has('colors_data')) {
                $data['colors_data'] = json_encode($request->colors_data);
            } else {
                $data['colors_data'] = null;
            }

    
            $this->product->storeProduct($data);
    
            DB::commit();
            return redirect()->route('admin.products.index')->with('success', 'Product has been added successfully.');
        } catch (\Exception $e) {
            dd($e->getMessage());
            DB::rollBack();
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function show($id)
    {
        $data = $this->product->findOrFail($id);
        return view('admin.products.show', compact('data'));
    }

    public function edit($id)
    {
        $data = $this->product->findOrFail($id);
        $form = [
            'type' => 'edit',
            'heading' => 'Edit Product',
            'method' => 'PUT',
            'action' => route('admin.products.update', $id),
            'cancel_url' => route('admin.products.index')
        ];
        
        $categories = Category::pluck('name', 'id');
        $subCategories = SubCategory::pluck('name', 'id');
        return view('admin.products.form', compact('data', 'form', 'categories', 'subCategories'));
    }

    public function update(Request $request, $id)
    {
        try {
            DB::beginTransaction();
    
            $data = $request->except(['_method', '_token', 'icon']);
    
            // Icon replacement
            if ($request->hasFile('icon')) {
                $data['icon'] = $request->file('icon')->store('products/icons', 'public');
            }
            
            if ($request->hasFile('size_chart')) {
                $data['size_chart'] = $request->file('size_chart')->store('products/icons', 'public');
            }
    
            // Colors handling
            // $data['colors'] = is_array($request->colors) ? $request->colors : explode(',', $request->colors);
    
            if ($request->has('colors_data')) {
                $data['colors_data'] = json_encode($request->colors_data);
            } else {
                $data['colors_data'] = null;
            }
            
            $this->product->whereUpdate(['id' => $id], $data);
    
            DB::commit();
            return redirect()->route('admin.products.index')->with('success', 'Product has been updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function destroy($id)
    {
        try {
            $this->product->findOrFail($id)->delete();
            return redirect()
                ->route('admin.products.index')
                ->with('success', 'Product has been deleted successfully.');
        } catch (\Exception $exception) {
            return redirect()
                ->back()
                ->with('error', $exception->getMessage());
        }
    }
}
