<?php

namespace App\Http\Controllers\Api;

use App\Models\Category;
use App\Models\Cashout;
use App\Models\ProductType;
use App\Models\StoreHour;
use App\Models\Card;
use App\Models\Concept;
use App\Models\Product;
use App\Models\User;
use App\Models\Ahtlete;
use Illuminate\Http\Request;
use App\Http\Controllers\Api\BaseController;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;



class ProductController extends BaseController
{
    public function getCategories()
    {
        return $this->respond(Category::all(), [], 200, 'Categories list fetched successfully');
    }
    
    public function getDashboard()
    {
        $services = Concept::with('category')->where('athlete_id',auth()->id())->whereNotNull('category_id')->get();
        return $this->respond(['total_revunue'=> 0,'total_cashout'=>0,'total_views'=>0,'services'=>$services], [], 200, 'Categories list fetched successfully');
    }

    public function getProductTypes(Request $request)
    {
        return $this->respond(ProductType::with('category','subCategory')->where('category_id',$request->category_id)->get(), [], 200, 'Product Types list fetched successfully');
    }

    public function storeProduct(Request $request)
    {
        $request->validate([
            'category_id' => 'required|exists:categories,id',
            'product_type_id' => 'required|exists:product_types,id',
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            // 'price' => 'required|numeric',
        ]);
        

        $product = Product::create([
            'athlete_id' => auth()->id(),
            'concept_id' => $request->concept_id,
            'category_id' => $request->category_id,
            'product_type_id' => $request->product_type_id,
            'name' => $request->name,
            'description' => $request->description,
            'price' => $request->price,
            'placement' => $request->placement,
             'colors' => json_encode($request->colors),
        ]);

        return $this->respond($product, [], 201, 'Product created successfully');
    }
    
     public function storeCashout(Request $request)
    {
        $request->validate([
            'amount' => 'required|numeric|min:1',
        ]);
    
        return DB::transaction(function () use ($request) {
            $user = Athlete::findOrFail(auth()->id());
    
            // Check if user has enough balance
            if ($request->amount > $user->total_sale) {
                return $this->respond(null, [], 400, 'Insufficient balance for cashout.');
            }
    
            // Create cashout
            $cashout = Cashout::create([
                'athlete_id' => $user->id,
                'amount'     => $request->amount,
            ]);
    
            // Deduct from total_sale
            $user->decrement('total_sale', $request->amount);
    
            return $this->respond($cashout, [], 201, 'Cashout created successfully');
        });
    }


    public function getMyProducts()
    {
        $products = Product::where('athlete_id', auth()->id())->with(['category', 'productType'])->get();
        return $this->respond($products, [], 200, 'Products list fetched successfully');
    }
    
     public function upsert(Request $request)
    {
        $storeHour = StoreHour::updateOrCreate(
            ['user_id' => auth()->id()],
            $request->only([
                'monday', 'tuesday', 'wednesday', 'thursday',
                'friday', 'saturday', 'sunday'
            ])
        );
        return $this->respond($storeHour, [], 201, 'Store hours saved successfully.');
    }
    
    public function storeCard(Request $request)
    {
        $storeHour = Card::updateOrCreate(
            ['user_id' => auth()->id()],
            $request->only([
                'card', 'expire_date', 'security_code', 'card_holder'
            ])
        );
        return $this->respond($storeHour, [], 201, 'Card Updated successfully.');
    }
    
    
     public function closeAccount(Request $request)
    {
        $user = Ahtlete::where('id', auth()->id())->update([
            'status' => 0,
            'close_account' => $request->close_account,
        ]);
    
        return $this->respond($user, [], 201, 'Account closed successfully.');
    }
    
    public function getStoreHours()
    {
        $storeHour = StoreHour::where('user_id', auth()->id())->first();

        if (!$storeHour) {
            return $this->respondNotFound('Store hours not found.');
        }

        return $this->respond($storeHour, [], 201, 'Store hours fetched successfully.');
    }
    
    public function getCard()
    {
        $storeHour = Card::where('user_id', auth()->id())->first();

        if (!$storeHour) {
            return $this->respondNotFound('Card not found.');
        }

        return $this->respond($storeHour, [], 201, 'Card fetched successfully.');
    }
    
    public function getAllProducts()
    {
        $products = Product::where('status', 'Accepted')->with(['category', 'productType'])->get();
        return $this->respond($products, [], 200, 'Products list fetched successfully');
    }
    
    public function getAllAthletes()
    {
        $products = Ahtlete::where('status', '1')->get();
        return $this->respond($products, [], 200, 'Athletes list fetched successfully');
    }
    
    public function getFuriousAthletes()
    {
        $products = Ahtlete::where('furious', '1')->get();
        return $this->respond($products, [], 200, 'Athletes list fetched successfully');
    }
    
    public function viewAthlete($id)
    {
        $products = Ahtlete::where('slug', $id)->first();
        return $this->respond($products, [], 200, 'Athlete fetched successfully');
    }
    
    public function allAthleteProducts($id)
    {
        $athlete = Ahtlete::where('slug', $id)->first();
        $products = Product::where('athlete_id',$athlete->id)->where('status', 'Accepted')->with(['category', 'productType','images'])->get();
        return $this->respond($products, [], 200, 'Products list fetched successfully');
    }
    
    public function productDetail($id)
    {
        $products = Product::where('id', $id)->with(['category', 'productType','images'])->get();
        return $this->respond($products, [], 200, 'Products Detail fetched successfully');
    }
    
     public function profile()
    {
        $profile = Ahtlete::where('id', auth()->id())->first();
        return $this->respond($profile, [], 201, 'Profile fetched successfully.');
    }

    public function updateProfile(Request $request)
    {
        $request->validate([
            'store_name' => 'nullable|string|max:255',
            'profile_picture' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'cover_photo' => 'nullable|image|mimes:jpeg,png,jpg|max:4096',
            'sport' => 'nullable|string|max:255',
            'social_media_reach' => 'nullable|string|max:255',
            'bio' => 'nullable|string|max:500',
            'description' => 'nullable|string|max:1000',
        ]);

        $athlete = Ahtlete::find(auth()->id());

        if (!$athlete) {
            return $this->respond(null, [], 404, 'Athlete not found.');
        }

        // Update fields
        $athlete->store_name = $request->store_name ?? $athlete->store_name;
        $athlete->sport = $request->sport ?? $athlete->sport;
        $athlete->social_media_reach = $request->social_media_reach ?? $athlete->social_media_reach;
        $athlete->bio = $request->bio ?? $athlete->bio;
        $athlete->description = $request->description ?? $athlete->description;

        // Handle profile picture
        if ($request->hasFile('profile_picture')) {
            $athlete->profile_picture = $request->file('profile_picture')->store('profile_pictures');
        }

        // Handle cover photo
        if ($request->hasFile('cover_photo')) {
            $athlete->cover_photo = $request->file('cover_photo')->store('cover_photos');
        }

        $athlete->save();

        return $this->respond($athlete, [], 200, 'Profile updated successfully.');
    }
    
    
    public function storeConcept(Request $request)
    {
        // Update fields
        $athlete = new Concept();
        $athlete->athlete_id = auth()->id();
        $athlete->category_id = $request->category_id;
        $athlete->content = $request->content;
        $athlete->selectedCategory = $request->selectedCategory;
        $athlete->description = $request->description;

        // Handle profile picture
        if ($request->hasFile('image')) {
            $paths = [];
            foreach ($request->file('image') as $file) {
                $paths[] = $file->store('profile_pictures');
            }
        
            // JSON ke form me save karo
            $athlete->image = json_encode($paths);
        }
        
        if ($request->hasFile('picture')) {
            $athlete->picture = $request->file('picture')->store('profile_pictures');
        }
        
        if ($request->hasFile('graphic_picture')) {
            $athlete->graphic_picture = $request->file('graphic_picture')->store('profile_pictures');
        }

        // Handle cover photo
        if ($request->hasFile('logo')) {
            $athlete->logo = $request->file('logo')->store('profile_pictures');
        }

        $athlete->save();

        return $this->respond($athlete, [], 200, 'Concept Store successfully.');
    }
    
    public function updateConcept(Request $request)
    {
        $athlete = Ahtlete::find(auth()->id());
        
        $concept = Concept::where('id',$request->concept_id)->where('athlete_id',auth()->id())->first();

        if($concept) {
            $concept->status = $request->status;
            $concept->save();
            return $this->respond($concept, [], 200, 'Already Submitted');
            
        }

        return $this->respond(null, [], 404, 'Concept not found.');
    }
    
    public function checkConcept(Request $request)
    {
        $athlete = Ahtlete::find(auth()->id());
        
        $concept = Concept::where('category_id',$request->category_id)->where('status','Completed')->where('athlete_id',auth()->id())->first();

        if($concept) {
            return $this->respond($concept, [], 200, 'Already Submitted');
        }

        return $this->respond(null, [], 404, 'Concept not found.');
    }
    
    
    public function accept(Request $request)
    {
        $athlete = Ahtlete::find(auth()->id());

        if (!$athlete) {
            return $this->respond(null, [], 404, 'Athlete not found.');
        }

        // Update fields
        $athlete->agree = 1;


        $athlete->save();

        return $this->respond($athlete, [], 200, 'Success');
    }
    
    
    public function agrement(Request $request)
    {
        $athlete = Ahtlete::find(auth()->id());
    
        if (!$athlete) {
            return $this->respond(null, [], 404, 'Athlete not found.');
        }
    
        // Validation
        $request->validate([
            'athlete_name' => [
                'required',
                Rule::unique('ahtletes', 'athlete_name')->ignore($athlete->id),
            ],
        ]);
    
        // Update fields
        $athlete->athlete_name = $request->athlete_name;
        $athlete->slug = Str::slug($request->athlete_name);
        $athlete->date = $request->date;
        $athlete->signature = $request->signature;
    
        $athlete->save();
    
        return $this->respond($athlete, [], 200, 'Success');
    }
}

