<?php

namespace App\Http\Controllers\Api;

use App\Models\User;
use App\Models\Cashout;
use App\Models\Concept;
use App\Models\BundleSubscription;
use App\Models\Order;
use App\Models\Ahtlete;
use App\Models\Product;
use App\Models\Category;
use App\Models\SubCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Api\BaseController;

class AdminUserController extends BaseController
{
    private $adminuser;

    public function __construct(User $adminuse)
    {
        $this->middleware('auth:user_api'); // agar admin ke liye alag API guard use kar rahe ho
        $this->user = $adminuse;
    }

    public function index()
    {
        try {
            $data = $this->user->all();
            return $this->respond($data, [], 200, 'Users fetched successfully');
        } catch (\Exception $e) {
            return $this->respond([], [], 500, $e->getMessage(), false);
        }
    }
    
    public function getDashboard()
    {
        try {
            $data['total_sale']      = Order::sum('total_price'); 
            $data['total_orders']    = Order::count();
            $data['total_customers'] = Order::count();
            $data['total_products']  = Product::count();

            $data['total_sale_percentage'] = 0.0;
            $data['days']['monday'] = 0;
            $data['days']['tuesday'] = 0;
            $data['days']['wednesday'] = 0;
            $data['days']['thursday'] = 0;
            $data['days']['friday'] = 0;
            $data['days']['saturday'] = 0;
            $data['days']['sunday'] = 0;
            return $this->respond($data, [], 200, 'Dashboard fetched successfully');
        } catch (\Exception $e) {
            return $this->respond([], [], 500, $e->getMessage(), false);
        }
    }
    
    public function getCashout()
    {
        try {
            $data = Cashout::orderBy('id','desc')->with('athlete')->get();
            return $this->respond($data, [], 200, 'Cashout fetched successfully');
        } catch (\Exception $e) {
            return $this->respond([], [], 500, $e->getMessage(), false);
        }
    }
    
    public function getConcept()
    {
        try {
            $data = Concept::where('status','Completed')->with('athlete')->get();
            return $this->respond($data, [], 200, 'Concept fetched successfully');
        } catch (\Exception $e) {
            return $this->respond([], [], 500, $e->getMessage(), false);
        }
    }
    
    public function getBundlesSubscription()
    {
        try {
            $data = BundleSubscription::with('user','package')->get();
            return $this->respond($data, [], 200, 'BundleSubscription fetched successfully');
        } catch (\Exception $e) {
            return $this->respond([], [], 500, $e->getMessage(), false);
        }
    }
    
    public function viewBundlesSubscription($id)
    {
        try {
            $data = BundleSubscription::with('user','package')->where('id',$id)->first();
            return $this->respond($data, [], 200, 'BundleSubscription fetched successfully');
        } catch (\Exception $e) {
            return $this->respond([], [], 500, $e->getMessage(), false);
        }
    }
    
    public function updateCashout(Request $request)
    {
        $request->validate([
            'id'          => 'required',
        ]);

        try {
            DB::beginTransaction();
            $cashout = Cashout::where('id',$request->id)->update(['status'=>'Completed']);
            DB::commit();
            return $this->respond($cashout, [], 201, 'Cashout Status Updated successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->respond([], [], 500, $e->getMessage(), false);
        }
    }

    public function store(Request $request)
    {
        $request->validate([
            'name'          => 'required|string|max:255',
            'email'   => 'required|unique:users'
        ]);

        try {
            DB::beginTransaction();

            $data = $request->except(['icon', 'size_chart']);
            $data['password'] = bcrypt($request->password);
            $data['email_verified_at'] = date('Y-m-d H:i:s');
            $user = $this->user->create($data);

            DB::commit();
            return $this->respond($user, [], 201, 'User added successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->respond([], [], 500, $e->getMessage(), false);
        }
    }

    public function show($id)
    {
        try {
            $data = $this->user->findOrFail($id);
            return $this->respond($data, [], 200, 'User details fetched');
        } catch (\Exception $e) {
            return $this->respond([], [], 404, $e->getMessage(), false);
        }
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'name'          => 'nullable|string|max:255'
        ]);

        try {
            DB::beginTransaction();

            $data = $request->except(['icon', 'size_chart']);


            $user = $this->user->findOrFail($id);
            $user->update($data);

            DB::commit();
            return $this->respond($user, [], 200, 'User updated successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->respond([], [], 500, $e->getMessage(), false);
        }
    }

    public function destroy($id)
    {
        try {
            $user = $this->user->findOrFail($id);
            $user->delete();

            return $this->respond([], [], 200, 'User deleted successfully');
        } catch (\Exception $e) {
            return $this->respond([], [], 500, $e->getMessage(), false);
        }
    }
    
    public function getAthletes()
    {
        try {
            $data = Ahtlete::with('card')->orderBy('id','desc')->get();
            return $this->respond($data, [], 200, 'Athletes fetched successfully');
        } catch (\Exception $e) {
            return $this->respond([], [], 500, $e->getMessage(), false);
        }
    }
    
    public function getAthletesProducts()
    {
        try {
            $data = Product::with('category','athlete','images')->orderBy('id','desc')->get();
            return $this->respond($data, [], 200, 'Products fetched successfully');
        } catch (\Exception $e) {
            return $this->respond([], [], 500, $e->getMessage(), false);
        }
    }
    
    public function viewAthlete($id)
    {
        try {
            $data = Ahtlete::where('id',$id)->first();
            return $this->respond($data, [], 200, 'Athlete fetched successfully');
        } catch (\Exception $e) {
            return $this->respond([], [], 500, $e->getMessage(), false);
        }
    }
    
    public function viewAthleteProduct($id)
    {
        try {
            $data = Product::with('category','athlete','images')->where('id',$id)->first();
            return $this->respond($data, [], 200, 'Product fetched successfully');
        } catch (\Exception $e) {
            return $this->respond([], [], 500, $e->getMessage(), false);
        }
    }
    
    public function updateStatusAthlete(Request $request, $id)
    {
        try {
            $request->validate([
                'status' => 'required' // Allowed statuses
            ]);
    
            $order = Ahtlete::findOrFail($id);
            $order->status = $request->status;
            $order->save();
    
            return $this->respond($order, [], 200, 'Athlete status updated successfully');
        } catch (\Illuminate\Validation\ValidationException $e) {
            return $this->respond([], $e->errors(), 422, 'Validation failed', false);
        } catch (\Exception $e) {
            return $this->respond([], [], 500, $e->getMessage(), false);
        }
    }
    
    public function updateFuriousAthlete(Request $request, $id)
    {
        try {
            $request->validate([
                'furious' => 'required|in:0,1' // Allowed values
            ]);
    
            // Agar request furious = 1 bheja gaya
            if ((int) $request->furious === 1) {
                $furiousCount = Ahtlete::where('furious', 1)->count();
    
                if ($furiousCount >= 5) {
                    return $this->respond([], [], 422, 'Maximum 5 athletes can be furious at a time', false);
                }
            }
    
            $athlete = Ahtlete::findOrFail($id);
            $athlete->furious = (int) $request->furious;
            $athlete->save();
    
            return $this->respond($athlete, [], 200, 'Athlete furious status updated successfully');
        } catch (\Illuminate\Validation\ValidationException $e) {
            return $this->respond([], $e->errors(), 422, 'Validation failed', false);
        } catch (\Exception $e) {
            return $this->respond([], [], 500, $e->getMessage(), false);
        }
    }

    
     public function updateStatusProduct(Request $request, $id)
    {
        try {
            $request->validate([
                'status' => 'required' // Allowed statuses
            ]);
    
            $order = Product::findOrFail($id);
            $order->status = $request->status;
            $order->save();
    
            return $this->respond($order, [], 200, 'Product status updated successfully');
        } catch (\Illuminate\Validation\ValidationException $e) {
            return $this->respond([], $e->errors(), 422, 'Validation failed', false);
        } catch (\Exception $e) {
            return $this->respond([], [], 500, $e->getMessage(), false);
        }
    }
    
    
   public function getOrders()
    {
        try {
            $data = Order::with([
                'items.product.athlete',
                'items.product.category'
            ])->orderBy('id', 'desc')->get();
    
            return $this->respond($data, [], 200, 'Orders fetched successfully');
        } catch (\Exception $e) {
            return $this->respond([], [], 500, $e->getMessage(), false);
        }
    }
    
    public function viewOrder($id) { try { $data = Order::with('items.product')->where('id',$id)->first(); return $this->respond($data, [], 200, 'Order fetched successfully'); } catch (\Exception $e) { return $this->respond([], [], 500, $e->getMessage(), false); } }
    
    public function updateStatus(Request $request, $id)
    {
        try {
            $request->validate([
                'status' => 'required'
            ]);
    
            $order = Order::with('product.athlete')->findOrFail($id); // load package as well
            $order->status = $request->status;
            $order->save();
    
            // If status = sent (case-insensitive)
            if (strtolower($request->status) === 'sent') {
                $athlete = $order->product->athlete;
    
                if ($athlete && $athlete->package) {
                    $badgeLevel = strtolower($athlete->badge_level);
                    $subscription = $athlete->has_active_subscription; 
                    // 👆 Package model me ek `type` column hona chahiye (e.g. 'pro' / 'standard')
    
                    $orderAmount = $order->total_price;
                    $commissionPercent = 0;
    
                    if ($subscription) {
                        // 🏅 Pro Tiers
                        if ($orderAmount >= 0 && $orderAmount <= 250) {
                            $commissionPercent = 10;
                        } elseif ($orderAmount <= 1000) {
                            $commissionPercent = 15;
                        } elseif ($orderAmount <= 2500) {
                            $commissionPercent = 20;
                        } elseif ($orderAmount <= 10000) {
                            $commissionPercent = 25;
                        } elseif ($orderAmount <= 20000) {
                            $commissionPercent = 30;
                        } else {
                            $commissionPercent = 40;
                        }
                    } else {
                        // 🟡 Standard Tiers
                        if ($orderAmount >= 0 && $orderAmount <= 250) {
                            $commissionPercent = 5;
                        } elseif ($orderAmount <= 1000) {
                            $commissionPercent = 8;
                        } elseif ($orderAmount <= 2500) {
                            $commissionPercent = 12;
                        } elseif ($orderAmount <= 10000) {
                            $commissionPercent = 15;
                        } elseif ($orderAmount <= 20000) {
                            $commissionPercent = 18;
                        } else {
                            $commissionPercent = 20;
                        }
                    }
    
                    // Commission calculate
                    $commissionAmount = ($orderAmount * $commissionPercent) / 100;
    
                    // Save in athlete (or in a separate commissions table)
                    $athlete->total_sale = ($athlete->total_sale ?? 0) + $commissionAmount;
                    $athlete->badge_amount = ($athlete->badge_amount ?? 0) + $commissionAmount;
                    $athlete->save();
                }
            }
    
            return $this->respond($order, [], 200, 'Order status updated successfully');
        } catch (\Illuminate\Validation\ValidationException $e) {
            return $this->respond([], $e->errors(), 422, 'Validation failed', false);
        } catch (\Exception $e) {
            return $this->respond([], [], 500, $e->getMessage(), false);
        }
    }


}
